# -*- coding: utf-8 -*-

# Copyright Martin Manns, Jason Sexauer
# Distributed under the terms of the GNU General Public License

# --------------------------------------------------------------------
# pyspread is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# pyspread is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with pyspread.  If not, see <http://www.gnu.org/licenses/>.
# --------------------------------------------------------------------

"""

exception_handling.py contains functions for handling exceptions generated by
user code

**Provides**

* :func:`get_user_codeframe`: Returns traceback that only includes the user's
                              execute code frames

"""

from types import TracebackType
from typing import Union


def get_user_codeframe(tb: TracebackType) -> Union[TracebackType, bool]:
    """Modify traceback to only include the user code's execution frame
    Always call in this fashion:

    .. code-block:: py

        e = sys.exc_info()
        user_tb = get_user_codeframe(e[2]) or e[2]

    so that you can get the original frame back if you need to
    (this is necessary because copying traceback objects is tricky and
    this is a good workaround)

    :param tb: Trace back information

    """
    while tb is not None:
        f = tb.tb_frame
        co = f.f_code
        filename = co.co_filename
        if filename[0] == '<':
            # This is a meta-descriptor
            # (probably either "<unknown>" or "<string>")
            # and is likely the user's code we're executing
            return tb
        else:
            tb = tb.tb_next
    # We could not find the user's frame.
    return False
